/****************************************************************
* Firmware Rev. 3 for the TI TRF7970A EVM
* 11.Nov.2013
* Copyright (C) 2010 Texas Instruments, Inc. 
*
* DESCRIPTION: The TRF797x is an integrated analog front end and
* data framing system for a 13.56 MHz RFID reader system.
* Built-in programming options make it suitable for a wide range
* of applications both in proximity and vicinity RFID systems.
* The reader is configured by selecting the desired protocol in
* the control registers. Direct access to all control registers
* allows fine tuning of various reader parameters as needed.
*
* The TRF797x is interfaced to a MSP430F2370 microcontroller
* through a parallel interface or a  SPI (serial) interface using
* hardware Usart. The MCU is the master device and initiates all
* communication with the reader. The interface is selected by
* jumper setting.
*
* The anti-collision procedures (as described in the ISO
* standards 14443A/B and 15693) are implemented in the MCU
* firmware to help the reader detect and communicate with one
* PICC/VICC among several PICCs/VICCs.
*
* Initial the Firmware is running in stand alone mode. Found tags
* are displayed by LEDs and corrisponding UID's are sent out via UART.
* If a host is detected, the Firmware will switch to remote mode
* by setting the remote_flag.
*
* The MCU communicates with the higher level host station
* (normally a PC) through a UART. The user can send the desired
* commands and the MCU interprets the data received and sends
* appropriate commands to the TRF796x.
* 
*
* AUTHOR:	Reiser Peter	DATE: 07 DEC 2010
* 			JD Crutchfield		  11 NOV 2013
*
* CHANGES:
* REV.	DATE		WHO	DETAIL
* 00	02Dec2010	RP	Orginal Version
* 01	07Dec2010	RP	Changed SPI clock frequency in spi.c
* 02	25Jun2013	Long	Change all code of iso14443a.c, iso14443a.h, iso14443b.c, iso14443b.h
* 03    11Nov2013   JD		Fixed UART communication stability and made slight modifications to Long's updates
* BUILT WITH:
* Code Composer Studio Core Edition Version: 5.4.0.00091
* (C) Copyright Texas Instruments, 2009. All rights reserved.
*****************************************************************
*****************************************************************
* All software and related documentation is provided "AS IS" and
* without warranty or support of any kind and Texas Instruments
* expressly disclaims all other warranties, express or implied,
* including, but not limited to, the implied warranties of
* merchantability and fitness for a particular purpose. Under no
* circumstances shall Texas Instruments be liable for any
* incidental, special or consequential damages that result from
* the use or inability to use the software or related
* documentation, even if Texas Instruments has been advised of
* the liability.
*
* Unless otherwise stated, software written and copyrighted by
* Texas Instruments is distributed as "freeware". You may use
* and modify this software without any charge or restriction.
* You may distribute to others, as long as the original author
* is acknowledged.
****************************************************************/

//===============================================================
// Program with hardware USART and paralel communication        ;
// interface with TRF reader chip.                              ;
//                                                              ;
// PORT4.0 - 4.7 - (IO0-IO7) for parallel interface             ;
//                           with reader chip                   ;
// PORT3.0 - PORT3.3 - USCI_B0 ---> SPI interface with reader   ;
// PORT3.4 - PORT3.5 - USCI_A0 ---> USB/UART control signals    ;
// PORT2.1 - IRQ                                                ;
// PORT3.3 - DATA_CLK                                           ;
// PORT1.7 - PORT1.3 - signaling LEDs                           ;
// PORT1.0 - TRF Enable                                         ;
//===============================================================

//#include <MSP430x23x0.h> 	// processor specific header
#include <stdlib.h>			// general purpose standard library
#include <stdio.h>			// standard input/output header
#include "host.h"
#include "iso14443a.h"
#include "iso14443b.h"
#include "iso15693.h"
#include "felica.h"
#include "mcu.h"
#include "trf797x.h"
#include "types.h"
#include "uart.h"

//===============================================================

u08_t buf[300];

u08_t enable = 0;

u08_t i_reg = 0x01;							// interrupt register

u08_t irq_flag = 0x00;
u08_t rx_error_flag = 0x00;
s08_t rxtx_state = 1;							// used for transmit recieve byte count

s16_t nfc_state;

u08_t remote_flag = 0;
u08_t stand_alone_flag = 1;
u08_t reader_mode = 0x00;					// determines how interrups will be handled

//===============================================================
//
//===============================================================

void main (void)	{
	
	// Settings
	WDTCTL = WDTPW + WDTHOLD;			// stop watchdog timer
	

	//new



	McuDelayMillisecond(2);				// wait until system clock started

	SLAVE_SELECT_PORT_SET;							// P3.0 - Slave Select
	SLAVE_SELECT_HIGH;

	#ifdef ENABLE_HOST
		UartSetup();					// settings for UART
	#endif

	McuDelayMillisecond(3);				// wait until system clock started

	ENABLE_SET;							// P1.0 is switched in output direction
	TRF_ENABLE;

	Trf797xCommunicationSetup();		// settings for communication with TRF

	McuDelayMillisecond(1);				// wait until system clock started
		
	Trf797xInitialSettings();			// Set MCU Clock Frequency to 6.78 MHz and OOK Modulation & reg 0x18 init
	
	// Now switch from DCO to external 6.78 MHz clock

	McuOscSel(0x00);       		// set the oscillator
	McuDelayMillisecond(10);
	
	// Re-configure the USART with this external clock

	Trf797xReConfig();
     	
	ENABLE_INTERRUPTS;					// General enable interrupts

	//LED_POWER_ON;						// board ist powered, green LED
    
	OOK_DIR_IN;							// set OOK port tristate
	enable = 1;							// indicates, that setting are done

	reader_mode = 0x00;
	stand_alone_flag = 1;				// stand alone mode
	remote_flag = 0;					// host is not active

	//set tag found LED port pin to output
	P1DIR |= BIT4;//output

	//Default Internal / External Antenna Select
	P1DIR |= BIT1;
	P1OUT |= BIT1; //internal antenna
	//P1OUT &= ~BIT1; //external antenna


	//init module IO ports
	P1DIR |= BIT2;//output
	P1OUT &= ~BIT2; //low-idle
	//P1OUT |= BIT2; //high
	P2DIR |= BIT4;//output
	P2OUT &= ~BIT4; //low-idle
	//P2OUT |= BIT4; //high
	P2DIR |= BIT5;//output
	P2OUT &= ~BIT5; //low-idle
	//P2OUT |= BIT5; //high



	//init unused module IO pins to outoput low
	P1DIR |= BIT3;//output
	P1OUT &= ~BIT3; //low-idle
	P1DIR |= BIT5;//output
	P1OUT &= ~BIT5; //low-idle
	P1DIR |= BIT6;//output
	P1OUT &= ~BIT6; //low-idle
	P1DIR |= BIT7;//output
	P1OUT &= ~BIT7; //low-idle
	P3DIR |= BIT7;//output
	P3OUT &= ~BIT7; //low-idle
	P3DIR |= BIT6;//output
	P3OUT &= ~BIT6; //low-idle
	P4DIR |= 0xFF;
	P4OUT &= ~0xFF;




    //	Settings done
    
	while(1)							// infinite loop
	{	
		#if TRIGGER						// in Mcu.h
			//LED_OPEN1_ON;
			LED_15693_ON;
			McuDelayMillisecond(1);
			//LED_OPEN1_OFF;
			LED_15693_OFF;
		#endif
		if(remote_flag == 1) 			// if in remote mode
		{								// remote mode can be disabled in host.h
			#ifdef ENABLE_HOST
				buf[4] = 0xff;			// "TRF7960 EVM" message in GUI
				stand_alone_flag = 0;
				//LED_ALL_OFF;
				LED_15693_OFF;
				//LED_POWER_ON;
				HostCommands();			// device works according host commands
			#endif
		}
		else
		{	
			#ifdef ENABLE15693				// this standard can be disabled in ISO15693.h
				Iso15693FindTag();			// detects ISO15693 in stand-alone mode
			#endif
			if(remote_flag == 0)
			{	
				#ifdef ENABLE14443A			// this standard can be disabled in ISO14443A.h 
					Iso14443aFindTag();		// detects ISO14443A in stand-alone mode
				#endif
			}
			if(remote_flag == 0)
			{
				#ifdef ENABLE14443B			// this standard can be disabled in ISO14443B.h
					Iso14443bFindTag();		// detects ISO14443B in stand-alone mode
				#endif
			}
			if(remote_flag == 0)
			{
				#ifdef ENABLE_FELICA		// this standard can be disabled in felica.h
					FindFelica();			// detects FeliCa in stand-alone mode
				#endif
			}
		}
	}
}
